<?php
/**
 * @package   Yoobox for YOOtheme Pro 0.0.3 build 1013.0701
 * @author    Yoobox https://www.yoobox.io
 * @copyright Copyright (C) Millians Solutions, SL
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

namespace Yoobox\Api;

class YooboxApi extends AbstractApi implements YooboxApiInterface
{
    protected string $apiEndpoint = 'https://yoobox.io/api';

    protected string $apiKey = '';

    protected ?string $siteOrigin = null;

    protected string $pluginVersion = '1.0.0';

    public function listAssets(array $args = []): array
    {
        return $this->get('assets', $args)['data'] ?? [];
    }

    public function loadAsset(string $id, array $args = []): array
    {
        return $this->get("assets/{$id}", $args)['data'] ?? [];
    }

    public function saveAsset(array $args = []): array
    {
        return $this->post('assets', $args)['data'] ?? [];
    }

    public function updateAsset(string $id, array $args = []): array
    {
        return $this->put("assets/{$id}", $args)['data'] ?? [];
    }

    public function withApiKey(string $key): self
    {
        $this->apiKey = $key;

        return $this;
    }

    public function withSiteOrigin(string $origin): self
    {
        $this->siteOrigin = $origin;

        return $this;
    }

    public function withPluginVersion(string $version): self
    {
        $this->pluginVersion = $version;

        return $this;
    }

    protected function getHeaders(): array
    {
        $headers = parent::getHeaders();

        if ($this->apiKey) {
            $headers += ['Authorization' => 'Bearer ' . $this->apiKey];
        }

        $headers += [
            'X-Yoobox' => $this->generateSignature(),
        ];

        return $headers;
    }

    /**
     * Generate signature for yoobox origin authentication.
     * Contains obfuscated origin and version information for abuse prevention
     * without exposing the actual origin domain.
     */
    protected function generateSignature(): string
    {
        if (!$this->siteOrigin) {
            return '';
        }

        // Create a consistent hash of the origin that can be used for rate limiting
        // but doesn't reveal the actual origin
        $originHash = hash('sha256', $this->siteOrigin . 'yoobox-salt-2025');
        $versionHash = hash('sha256', $this->pluginVersion . 'yoobox-version-salt-2025');

        // Include timestamp for basic freshness (but not too strict)
        $hourlyTimestamp = floor(time() / 3600); // Changes every hour

        // Generate signature with obfuscated data
        $signatureData = $originHash . '|' . $versionHash . '|' . $hourlyTimestamp;
        $signature = hash('sha256', $signatureData . 'yoobox-plugin-signature');

        // Return in recognizable format: yb_ + origin_identifier + version_identifier + time_signature
        return 'yb_' .
            substr($originHash, 0, 12) . '_' .
            substr($versionHash, 0, 8) . '_' .
            substr($signature, 0, 16);
    }
}
