<?php
/**
 * @package   Yoobox for YOOtheme Pro 0.0.3 build 1013.0701
 * @author    Yoobox https://www.yoobox.io
 * @copyright Copyright (C) Millians Solutions, SL
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

namespace Yoobox\Api;

use YOOtheme\Http\Response;
use YOOtheme\HttpClientInterface;

abstract class AbstractApi
{
    protected string $apiEndpoint = '';

    protected HttpClientInterface $client;

    public function __construct(HttpClientInterface $client)
    {
        $this->client = $client;
    }

    public function withEndpoint(string $endpoint): self
    {
        $this->apiEndpoint = $endpoint;

        return $this;
    }

    protected function getUrl(string $name): string
    {
        return $this->apiEndpoint ? "{$this->apiEndpoint}/{$name}" : $name;
    }

    protected function getHeaders(): array
    {
        return [
            'Accept' => 'application/json',
            'Content-Type' => 'application/json',
        ];
    }

    public function get(string $name, array $args = []): array
    {
        return $this->request('GET', $name, $args);
    }

    public function post(string $name, array $args = []): array
    {
        return $this->request('POST', $name, $args);
    }

    public function put(string $name, array $args = []): array
    {
        return $this->request('PUT', $name, $args);
    }

    public function delete(string $name, array $args = []): array
    {
        return $this->request('DELETE', $name, $args);
    }

    protected function request(string $method, string $name, array $args): array
    {
        $url = $this->getUrl($name);
        $headers = $this->getHeaders();

        switch ($method) {
            case 'GET':
                $query = $this->buildQuery($url, $args);
                $response = $this->client->get("{$url}{$query}", compact('headers'));

                break;

            case 'POST':
                $response = $this->client->post($url, json_encode($args), compact('headers'));

                break;

            case 'PUT':
                $response = $this->client->put($url, json_encode($args), compact('headers'));

                break;

            case 'PATCH':
                $response = $this->client->patch($url, json_encode($args), compact('headers'));

                break;

            case 'DELETE':
                $query = $this->buildQuery($url, $args);
                $response = $this->client->delete("{$url}{$query}", compact('headers'));

                break;

            default:
                throw new \Exception("Call to undefined method {$method}");
        }

        return $this->processResponse($response, $url);
    }

    protected function buildQuery($url, array $args): string
    {
        $query = parse_url($url)['query'] ?? '';
        $query = ($query ? '&' : '?') . http_build_query($args, '', '&');

        return $query;
    }

    protected function processResponse(Response $response): array
    {
        $result = json_decode($response->getBody(), true);
        $success = $response->getStatusCode() >= 200 && $response->getStatusCode() <= 299;

        if (!$success) {
            $code = $response->getStatusCode() ?? 400;
            $message = $result['message'] ?? $response->getReasonPhrase() ?? 'Unknown Error';

            throw new \Exception($message, $code);
        }

        return $result ?? [];
    }
}
