<?php
/**
 * @package   Yoobox for YOOtheme Pro 0.0.3 build 1013.0701
 * @author    Yoobox https://www.yoobox.io
 * @copyright Copyright (C) Millians Solutions, SL
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

defined('_JEXEC') or die();

use Joomla\CMS\Factory;

abstract class plgSystemYooboxHelper
{
    const MIN_PHP_VERSION = '8.1';
    const MIN_JOOMLA_VERSION = '4.0';
    const MIN_YTP_VERSION = '4.5.0';
    const MAX_YTP_VERSION = '4.5.99';
    const BLOCK_YTP_UPDATE = 'YES';

    public static function adminNotice(string $message): void
    {
        $app = Factory::getApplication();

        if ($app->isClient('administrator')) {
            $app->enqueueMessage($message, 'warning');
        }
    }

    public static function validatePlatform()
    {
        if (version_compare(PHP_VERSION, self::MIN_PHP_VERSION, 'lt')) {
            throw new RuntimeException(
                sprintf(
                    'Yoobox plugin requires a more recent version of PHP, please update PHP to v%s or later.',
                    self::MIN_PHP_VERSION
                )
            );
        }

        if (version_compare(JVERSION, self::MIN_JOOMLA_VERSION, 'lt')) {
            throw new RuntimeException(
                sprintf(
                    'Yoobox plugin requires a more recent version of Joomla, please update Joomla to v%s or later.',
                    self::MIN_JOOMLA_VERSION
                )
            );
        }

        $theme = @simplexml_load_file(JPATH_ROOT . '/templates/yootheme/templateDetails.xml');
        $themeVersion = (string) ($theme->version ?? '');

        if ($theme === null || version_compare($themeVersion, self::MIN_YTP_VERSION, 'lt')) {
            throw new RuntimeException(
                sprintf(
                    'Yoobox plugin requires a more recent version of YOOtheme Pro, please update YOOtheme Pro to v%s or later.',
                    self::MIN_YTP_VERSION
                )
            );
        }

        if (version_compare($themeVersion, self::MAX_YTP_VERSION, 'gt')) {
            throw new RuntimeException(
                sprintf(
                    'The currently installed Yoobox plugin does not support YOOtheme Pro v%s, try updating Yoobox to its latest version or contact support for further help.',
                    $themeVersion
                )
            );
        }
    }

    // checks if the theme being installed is compatible with the plugin
    public static function preinstallThemeCheck(SimpleXMLElement $manifest): void
    {
        if ((string) $manifest->packagename !== 'yootheme') {
            return;
        }

        $version = (string) $manifest->version;

        if (
            version_compare($version, self::MIN_YTP_VERSION, 'lt') ||
            (self::BLOCK_YTP_UPDATE === 'YES' && version_compare($version, self::MAX_YTP_VERSION, 'gt'))
        ) {
            throw new RuntimeException(
                sprintf(
                    'YOOtheme Pro v%s is not supported by the currently installed Yoobox plugin, disable Yoobox before retrying.',
                    $version
                )
            );
        }
    }

    public static function validateChecksums(bool $force = false): void
    {
        $root = JPATH_ROOT . '/plugins/system/yoobox';
        $checksums = "$root/checksums";
        $checksumsPass = "$root/checksums.pass";

        if (!$force && file_exists($checksumsPass)) {
            return;
        }

        $file = file_exists($checksums) ? fopen($checksums, 'r') : null;

        // skip if checksum file is missing
        if (!$file) {
            return;
        }

        $log = [];
        while ($row = fgets($file)) {
            list($md5, $fileName) = explode(' ', trim($row), 2);

            $filePath = $root . '/' . trim($fileName);

            if (!file_exists($filePath)) {
                $log['missing'][] = $filePath;

                continue;
            }

            $fileMd5 = md5_file($filePath);

            if ($fileMd5 !== $md5) {
                $log['changed'][] = $filePath;
            }
        }

        if (empty($log)) {
            ($file = fopen($checksumsPass, 'w')) or die('Unable to create file!');
            fwrite($file, 'OK');
            fclose($file);
        }

        if (!empty($log)) {
            $message = 'Yoobox plugin execution has been prevented due to corrupted installation or altered files. Please reinstall Yoobox.';
            $message .= '<br>';

            foreach ($log as $type => $files) {
                $type = strtoupper($type);
                foreach ($files as $file) {
                    $message .= "<br>$type - $file";
                }
            }

            throw new RuntimeException($message);
        }
    }

    public static function clearCache($files = null)
    {
        $files = $files ?: self::getCacheFiles();

        foreach ($files as $file) {
            if (is_iterable($file)) {
                self::clearCache($file);
            } elseif ($file->isFile()) {
                unlink($file->getRealPath());
            } elseif ($file->isDir()) {
                rmdir($file->getRealPath());
            }
        }
    }

    private static function getCacheFiles()
    {
        $files = [new SplFileInfo(JPATH_ROOT . '/templates/yootheme/cache/schema-1.gql')];

        $cachePath = JPATH_ROOT . '/templates/yootheme/cache/yoobox';

        if (file_exists($cachePath)) {
            $iterator = new RecursiveDirectoryIterator($cachePath, FilesystemIterator::SKIP_DOTS);
            $files[] = new RecursiveIteratorIterator($iterator, RecursiveIteratorIterator::CHILD_FIRST);
        }

        return $files;
    }
}
